package bunshin;


import bunshin.listeners.*;
import bunshin.storage.*;

import java.net.URL;
import java.util.Hashtable;
import java.util.Collection;
import rice.p2p.commonapi.*;


/**
 *
 * This interface is exported by all instances of Bunshin DHT. 
 * 
 * @author Ruben Mondejar
 */
public interface Bunshin {

  /**
   * Stores the (key,value) pair in the Bunshin DHT in the default context and with default field
   *
   * @param key Id of the value
   * @param value Object to store
   */
  public void put(Id key, Object value);
  
  /**
   * Stores the (key,value,field) in the Bunshin DHT in the default context
   *
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   */  
  public void put(Id key, Object value, String field);
  
  /**
   * Stores the (context,key,value) in the Bunshin DHT with default field
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   */  
  public void put(String context, Id key, Object value);

  /**
   * Stores the (context,key,value,field) in the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   */  

  public void put(String context, Id key, Object value, String field);
  
  /**
   * Stores the (context,key,value,field,pass) in the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param pass password
   */  
  public void putSecure(String context, Id key, Object value, String field, String pass);
  
  /**
   * Stores the (context,key,value,field,pass) in the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param pass password
   * @param getcheck if true, the password will be only for remove 
   */  
  public void putSecure(String context, Id key, Object value, String field, String pass, boolean getcheck);
  
    
  
  /**
   * Stores the (key,value) pair in the Bunshin DHT in a default field
   *
   * @param key Id of the value
   * @param value Object to store
   * @param client listener BunshinPutClient
   */
  public void put(Id key, Object value, BunshinPutClient client);
  
  /**
   * Stores the (key,value,field) in the Bunshin DHT with result listener
   *
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param client listener BunshinPutClient
   */  
  public void put(Id key, Object value, String field, BunshinPutClient client);


   /**
   * Stores the (context,key,value) in the Bunshin DHT with result listener
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param client listener BunshinPutClient
   */  
  
  public void put(String context, Id key, Object value, BunshinPutClient client);

   /**
   * Stores the (context,key,value,field) in the Bunshin DHT with result listener
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param client listener BunshinPutClient
   */  
  
  public void put(String context, Id key, Object value, String field, BunshinPutClient client);
  
    /**
   * Stores the (context,key,value,field,pass) in the Bunshin DHT with result listener
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param pass password
   * @param client listener BunshinPutClient
   */ 
  public void putSecure(String context, Id key, Object value, String field, String pass, BunshinPutClient client);
  
  /**
   * Stores the (context,key,value,field,pass) in the Bunshin DHT with result listener
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param pass password
   * @param client listener BunshinPutClient
   * @param getcheck if true, the password will be only for remove 
   */  
  
  public void putSecure(String context, Id key, Object value, String field, String pass, boolean getcheck, BunshinPutClient client);

  

  /**
   * Stores the URL and this content (key,url,field) in the Bunshin DHT and the value is downloaded, in the default context.
   *
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   */
   public void putURL(Id key, URL url,String field);
  
  /**
   * Stores the URL and this content (key,url,field) in the Bunshin DHT and the value is downloaded.
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   */
   public void putURL(String context, Id key, URL url, String field);
	
	
  /**
   * Modifies the content of a specific field
   *
   * @param key Id of the value
   * @param value Object to store
   * @param field String key
   * @param add if true for insertion mode, and false to erase mode
   */ 	
   public void modify(Id key, Object value, String field, boolean add);
   
     /**
   * Modifies the content of a specific field
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param value Object to modify
   * @param field String key
   * @param add if true for insertion mode, and false to erase mode
   */
   public void modify(String context, Id key, Object value, String field, boolean add);
   
     /**
   * Modifies the content of a specific field
   *
   * @param key Id of the value
   * @param values Collection to modify
   * @param field String key
   * @param add if true for insertion mode, and false to erase mode
   */
   public void modify(Id key, Collection values, String field, boolean add);

  /**
   * Modifies the content of a specific field
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param values Collection to modify
   * @param field String key
   * @param add if true for insertion mode, and false to erase mode
   */
   public void modify(String context,Id key, Collection list, String field, boolean add);
    

   /**
   * Removes the key/value of the Bunshin DHT 
   *
   * @param key Id of the value   
   */  
  public void remove(Id key); 

   /**
   * Removes the (context,key)/value of the Bunshin DHT 
   *
   * @param key Id of the value   
   * @param field String key   
   */  
  public void remove(Id key, String field);
   /**
   * Removes the (context,key)/value of the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value   
   */  
  public void remove(String context, Id key);

   /**
   * Removes the (context,key,field)/value of the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key   
   */  
  public void remove(String context,Id key, String field);
  
   /**
   * Removes the (context,key,field,pass)/value of the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key
   * @param pass password   
   */  
  public void removeSecure(String context,Id key, String field, String pass);
    
  /**
   * Removes the key/value of the Bunshin DHT 
   *
   * @param key Id of the value
   * @param client listener BunshinRemoveClient   
   */  
  public void remove(Id key, BunshinRemoveClient client);

   /**
   * Removes the (context,key)/value of the Bunshin DHT 
   *
   * @param key Id of the value   
   * @param field String key
   * @param client listener BunshinRemoveClient   
   */  
  public void remove(Id key, String field, BunshinRemoveClient client);
   /**
   * Removes the (context,key)/value of the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value   
   * @param client listener BunshinRemoveClient
   */  
  public void remove(String context, Id key, BunshinRemoveClient client);

   /**
   * Removes the (context,key,field)/value of the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key   
   * @param client listener BunshinRemoveClient
   */  
  public void remove(String context,Id key, String field, BunshinRemoveClient client);
  
   /**
   * Removes the (context,key,field,pass)/value of the Bunshin DHT 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key
   * @param pass password
   * @param client listener BunshinRemoveClient   
   */  
  public void removeSecure(String context,Id key, String field, String pass, BunshinRemoveClient client);

  /**
   * Returns the value of the specific key if the values is stored in the local node
   *
   * @param key Id of the value   
   */   
  public Object getLocal(Id key);
  
  /**
   * Returns the value of the specific key if the values is stored in the local node
   *
   * @param key Id of the value
   * @param field String key   
   */   
  public Object getLocal(Id key, String field);
  
  /**
   * Returns the value of the specific key if the values is stored in the local node
   *
   * @param context Id of the context
   * @param key Id of the value
   * 
   */   
  public Object getLocal(String context, Id key);
  
  /**
   * Returns the value of the specific key if the values is stored in the local node
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key
   * 
   */   
  public Object getLocal(String context, Id key, String field);
  
 
  /**
   * Returns the value of the specific key if the values is stored in the local node
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key
   * @param pass password   
   * 
   */   
  public Object getLocalSecure(String context, Id key, String field, String pass);
    
  /**
   * Returns the value of the specific key using the Client application 
   *
   * @param key Id of the value 
   * @param client listener that implements BunshinClient     
   */   
  public void get(Id key, BunshinGetClient client);
  
  /**
   * Returns the values of the specific key and sets the Client application 
   *
   * @param key Id of the value
   * @param field String key
   * @param client listener that implements BunshinClient     
   */    
  public void get(Id key, String field, BunshinGetClient client);
  
  /**
   * Returns the values of the specific key and sets the Client application 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param client listener that implements BunshinClient     
   */  
  public void get(String context,Id key, BunshinGetClient client);

  /**
   * Returns the values of the specific key and sets the Client application 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key
   * @param client listener that implements BunshinClient     
   */  
  public void get(String context,Id key, String field, BunshinGetClient client);
    
  /**
   * Returns the values of the specific key and sets the Client application 
   *
   * @param context Id of the context
   * @param key Id of the value
   * @param field String key
   * @param pass password  
   * @param client listener that implements BunshinClient     
   */  
  public void getSecure(String context,Id key, String field, String pass, BunshinGetClient client);
    
   /**
   * Returns the URL of the specific key 
   *
   * @param key Id of the value
   * @param field String key
   * @param client listener that implements BunshinClient  
   */
  
  public void getURL(Id key, String field, BunshinURLClient client);
  
  /**
   * Returns the URL of the specific key
   *
   * @param context Id of the context
   * @param key Id of the value   
   * @param field String key
   * @param client listener that implements BunshinClient  
   */
  public void getURL(String context, Id key, String field, BunshinURLClient client);
  
  
   
  /**
   * Before leaving the network, the application must call to this method
   */
  public void leave();

  
   /**
   * Sets the listener of the specific key and this notifies about the incoming links 
   *   
   * @param id Id of the value      
   * @param listener that implements RemoteListener listener 
   * @param field String key
   */
  public void setRemoteListener(Id id, RemoteListener listener, String field);

   /**
   * Sets the listener of the specific key and this notifies about the incoming links 
   *   
   * @param context Id of the context
   * @param id Id of the value      
   * @param listener that implements RemoteListener listener 
   * @param field String key
   */
  public void setRemoteListener(String context,Id id, RemoteListener listener, String field);

   /**
   * Removes the listener of the specific key 
   *   
   * @param id Id of the value         
   * @param field String key
   */
  public void removeRemoteListener(Id id, String field);

   /**
   * Removes the listener of the specific key 
   *   
   * @param context Id of the context
   * @param id Id of the value         
   * @param field String key
   */
  public void removeRemoteListener(String context, Id id, String field);

  
  /**
   * Sets the number of replicas  
   *
   * @param the number of replicas 
   */
  public void setReplicationFactor(int replicas);
  
  /**
   * Sets the storage manager
   *
   * @param storage manager
   */  
   public void setStorageManager(StorageManager storage);
 	
  /**
   * Returns the number of replicas used 
   *
   * @return the number of replicas 
   */
  public int getReplicationFactor();
  
  public void setRefreshTime(int time);
  
  public void setEndPoint (Endpoint ep);
   
  public StorageManager getStorageManager();
  
  public void activateCache();
  
  public void activateDebug();
  
  public Hashtable getStorageInfo();
  
  public Hashtable getReplicaInfo();
  
	   /**
   * Sets the information of a specific context
   *   
   * @param context Id of the context
   * @param path String path of the storage directory (DiskStorage and XMLStorage) 
   * @param url String url base (ex : http://www.infourl.com)
   * @param extra paths of class/jar for classLoader
   */
    public void setInfoContext(String context, String path, String url, URL[] URLsList);


  }

